# -*- coding: utf-8 -*-
"""
config.py — Pytheia100 v1.1.0 (mirrors core's pointing_model verbatim)
Only edit this file. Do NOT modify protected files:
  Pytheia100_main.py, Pytheia100_core.py, pytheia_guard.py, INTEGRITY.json.
"""

import sys
import numpy as np

# ---------------------------------------------------------------------
# Fallback for colored text (avoid extra dependency)
# ---------------------------------------------------------------------
try:
    from termcolor import colored  # optional
except Exception:
    def colored(s, *_args, **_kwargs):
        return str(s)

# ---------------------------------------------------------------------
# User settings
# ---------------------------------------------------------------------
# Data / I/O
input_dat  = "Ecam_astrometry_pm-02-04-2025.dat"   # input_dat or DATA_FILE
DATA_FILE  = input_dat
output_dir = "My_Pointing_model_results"
OUTPUT_DIR = output_dir

# Mode: "azel" or "hadec"
mode = "azel"         # switch here when you need HA/Dec
MODE = mode           # alias for compatibility

# Site latitude (degrees) — used to compute phi in HA/Dec formulas
lat = -29.259
phi = np.radians(lat)  # radians

# Default term selection (order matters)
selected_terms = ["AN", "AW", "IA", "IE", "CA", "TX", "ECES", "HZSZ2"]

# Optional auxiliaries (scalar or arrays) — can be left as None
aux1 = None
aux2 = None

# Plot/report knobs (picked up if present)
CORRELATION_THRESHOLD = 0.8
PLOT_FIGSIZE = (18, 18)
PLOT_DPI = 300
PLOT_STYLE = "default"

# ---------------------------------------------------------------------
# Pointing model 
# ---------------------------------------------------------------------
def pointing_model(coord1, coord2, selected_terms, aux1, aux2):
    coord1 = np.asarray(coord1)
    coord2 = np.asarray(coord2)
    n = len(coord1)

    az = ha = coord1
    el = dec = coord2
    Z = 0.5 * np.pi - el if el is not None else None

    if aux1 is None:
        aux1 = np.zeros(n)
    elif np.isscalar(aux1):
        aux1 = np.full(n, aux1)
    else:
        aux1 = np.asarray(aux1)

    if aux2 is None:
        aux2 = np.zeros(n)
    elif np.isscalar(aux2):
        aux2 = np.full(n, aux2)
    else:
        aux2 = np.asarray(aux2)

    sin, cos, tan, sec = np.sin, np.cos, np.tan, lambda x: 1 / np.cos(x)

    sin_az, cos_az = sin(az), cos(az)
    sin_el, cos_el = sin(el), cos(el)
    tan_el, sec_el = tan(el), sec(el)
    sin_ha, cos_ha = sin(ha), cos(ha)
    sin_dec, cos_dec = sin(dec), cos(dec)
    tan_dec, sec_dec = tan(dec), sec(dec)

    cos_z  = cos(Z)
    cos_2z = cos(2 * Z)
    cos_4z = cos(4 * Z)
    sin_z  = sin(Z)
    sin_2z = sin(2 * Z)
    sin_3z = sin(3 * Z)
    sin_4z = sin(4 * Z)
    pow_3z = Z**3
    pow_5z = Z**5

    az_terms = {
        "IA":   np.ones(n),
        "NPAE": tan_el,
        "CA":   sec_el,
        "AN":   sin_az * tan_el,
        "AW":   cos_az * tan_el,
        "HSCA": -cos_az / cos_el,
        "HVCA": -cos_az * tan_el,
        "ACEC": -cos_az,
        "ACES":  sin_az,
        "HACA1": cos_az,
        "HSCZ1": cos_z / cos_el,
        "HSCZ2": cos_2z / cos_el,
        "HSCZ4": cos_4z / cos_el,
        "HSSA1": sin_az / cos_el,
        "HSSZ1": sin_z / cos_el,
        "HSSZ2": sin_2z / cos_el,
        "HSSZ3": sin_3z / cos_el,
        "NRX":   np.ones(n),
        "NRY":   tan_el,
        "AUX1A": aux1,
        "AUX2A": aux2,
        "AUX1S": aux1 * sec_el,
        "AUX2S": aux2 * sec_el,
    }

    el_terms = {
        "IE":   -np.ones(n),
        "TF":    cos_el,
        "TX":    1 / tan_el,
        "AN":    cos_az,
        "AW":   -sin_az,
        "ECEC":  cos_el,
        "ECES":  sin_el,
        "HZCZ1": cos_z,
        "HZCZ2": cos_2z,
        "HZCZ4": cos_4z,
        "HZSA":  sin_az,
        "HZSA2": sin(2 * az),
        "HZSZ1": sin_z,
        "HZSZ2": sin_2z,
        "HZSZ4": sin_4z,
        "PZZ3":  pow_3z,
        "PZZ5":  pow_5z,
        "NRX":  -sin_el,
        "NRY":   cos_el,
        "AUX1E": aux1,
        "AUX2E": aux2,
    }

    h_terms = {
        "IH":   np.ones(n),
        "CH":   sec_dec,
        "MA":  -cos_ha * tan_dec,
        "ME":   sin_ha * tan_dec,
        "NP":   tan_dec,
        "TF":   np.cos(phi) * sin_ha * sec_dec,
        "TX":  (np.cos(phi) * sin_ha) /
               ((sin_dec * np.sin(phi) + cos_dec * cos_ha * np.cos(phi)) * cos_dec),
        "AUX1H": aux1,
        "AUX1X": aux1 * sec_dec,
        "AUX2H": aux2,
        "AUX2X": aux2 * sec_dec,
        "FO":    cos_ha,
        "HCEC":  cos_ha,
        "HCES":  sin_ha,
        "X2HC":  np.cos(2 * ha) * sec_dec,
        "HXCH":  np.cos(ha) * sec_dec,
        "HXSH2": np.sin(2 * ha) * sec_dec,
    }

    dec_terms = {
        "ID":   np.ones(n),
        "MA":   sin_ha,
        "ME":   cos_ha,
        "TF":  (np.cos(phi) * cos_ha * sin_dec - np.sin(phi) * cos_dec),
        "TX":  (np.cos(phi) * cos_ha * sin_dec - np.sin(phi) * cos_dec) /
               ((sin_dec * np.sin(phi) + cos_dec * cos_ha * np.cos(phi)) * cos_dec),
        "DCEC":  cos_dec,
        "DCES":  sin_dec,
        "AUX1D": aux1,
        "AUX2D": aux2,
        "HDSH2": np.sin(2 * ha),
    }

    azel_terms_set  = set(az_terms) | set(el_terms)
    hadec_terms_set = set(h_terms)  | set(dec_terms)
    selected_set    = set(selected_terms)

    if mode == "azel" and not selected_set <= azel_terms_set:
        print(colored(f"\nERROR: Invalid terms in Az/El mode: {selected_set - azel_terms_set}", "red"))
        print(colored("Please check your terms and try again.", "red"))
        sys.exit(1)
    elif mode == "hadec" and not selected_set <= hadec_terms_set:
        print(colored(f"\nERROR: Invalid terms in HA/Dec mode: {selected_set - hadec_terms_set}", "red"))
        print(colored("Please check your terms and try again.", "red"))
        sys.exit(1)

    X_rows = []
    term_list = []

    for term in selected_terms:
        if mode == "azel":
            az_col  = az_terms.get(term, np.zeros(n))
            el_col  = el_terms.get(term, np.zeros(n))
        else:
            az_col  = h_terms.get(term, np.zeros(n))
            el_col  = dec_terms.get(term, np.zeros(n))

        full_col = np.concatenate([az_col, el_col])
        X_rows.append(full_col)
        term_list.append(term)

    X = np.column_stack(X_rows) if X_rows else np.zeros((2 * n, 0))
    return X, term_list
